# Backend Integration Guide - Maya Eslami Trading Platform

This document outlines the synchronized data structure and backend integration plan for the Maya Eslami trading platform.

## ✅ Completed Synchronization

### Admin Interface ↔ Frontend Template Alignment

All admin sections have been synchronized with the frontend template:

1. **Content Management**: Now handles trading-specific categories
2. **Signal Management**: Aligned with actual trading signal requirements
3. **Field Consistency**: All forms match expected data structures

## 📊 Data Models & Schemas

### Core Entities

#### 1. Trading Signals (`models/schemas.js` - TradingSignalSchema)
```javascript
{
  id: string (required),
  type: 'buy' | 'sell' (required),
  market: 'forex' | 'crypto' | 'gold' | 'stock' (required),
  symbol: string (required), // EUR/USD, BTC/USDT, XAU/USD
  entryPrice: number (required),
  stopLoss: number (required),
  takeProfit: number (required),
  analysis: string (optional),
  status: 'active' | 'completed' | 'cancelled',
  telegramChannel: string (required),
  createdAt: date,
  sentAt: date,
  members: number,
  result: {
    profit: number,
    closePrice: number,
    closeTime: date
  }
}
```

#### 2. Content/Articles (`models/schemas.js` - ContentSchema)
```javascript
{
  id: string (required),
  title: string (required),
  slug: string (required),
  category: enum [
    'آموزش تحلیل تکنیکال',
    'استراتژی معاملاتی',
    'اخبار بازار مالی',
    'مدیریت ریسک',
    'احساسات بازار',
    'تحلیل فارکس',
    'تحلیل کریپتو',
    'تحلیل طلا',
    'اخبار اقتصادی',
    'راهنمای تریدینگ'
  ],
  summary: string (required),
  content: string (required),
  status: 'draft' | 'published' | 'archived',
  views: number,
  author: string (required),
  publishedAt: date,
  createdAt: date,
  updatedAt: date
}
```

#### 3. Users (`models/schemas.js` - UserSchema)
```javascript
{
  id: string (required),
  name: string (required),
  email: string (required, unique),
  password: string (required, hashed),
  role: 'user' | 'premium' | 'admin',
  subscription: {
    type: 'free' | 'monthly' | 'yearly',
    startDate: date,
    endDate: date,
    status: 'active' | 'expired' | 'cancelled'
  },
  telegramUserId: string,
  preferences: {
    language: string,
    notifications: boolean,
    markets: array[string]
  },
  createdAt: date,
  lastLogin: date,
  isActive: boolean
}
```

#### 4. Market Data (`models/schemas.js` - MarketDataSchema)
```javascript
{
  id: string (required),
  symbol: string (required), // USD/IRR, EUR/USD, XAU/USD, BTC/USDT
  name: string (required),
  type: 'currency' | 'gold' | 'crypto' | 'stock',
  price: number (required),
  change: number,
  changePercent: number,
  high24h: number,
  low24h: number,
  volume: number,
  lastUpdate: date,
  source: string
}
```

## 🔗 API Endpoints

### Available Routes (`routes/api.js`)

#### Trading Signals
- `GET /api/signals` - Get all trading signals
- `POST /api/signals` - Create new trading signal
- `PUT /api/signals/:id` - Update trading signal
- `DELETE /api/signals/:id` - Delete trading signal

#### Content Management
- `GET /api/content` - Get all articles (with filtering)
- `POST /api/content` - Create new article
- `PUT /api/content/:id` - Update article
- `DELETE /api/content/:id` - Delete article

#### User Management
- `GET /api/users` - Get all users
- `POST /api/users` - Create new user
- `PUT /api/users/:id` - Update user
- `DELETE /api/users/:id` - Delete user

#### Market Data
- `GET /api/market-data` - Get current market data
- `GET /api/market-data?type=currency` - Get specific data type

#### Settings
- `GET /api/settings` - Get all settings
- `PUT /api/settings` - Update settings

#### Analytics
- `GET /api/analytics/dashboard` - Get dashboard analytics

## 🗄️ Database Options

### Current: File-based Storage (`config/database.js`)
- Using JSON files for development
- Located in `public/api/` directory
- Includes DatabaseAdapter for abstraction

### Future Options:

#### MongoDB Configuration
```javascript
{
  url: 'mongodb://localhost:27017/maya-eslami',
  collections: {
    users: 'users',
    signals: 'trading_signals',
    content: 'content',
    marketData: 'market_data',
    sentiment: 'sentiment_data',
    news: 'news',
    settings: 'settings',
    telegramBots: 'telegram_bots'
  }
}
```

#### PostgreSQL Configuration
```javascript
{
  host: 'localhost',
  port: 5432,
  database: 'maya_eslami',
  username: 'postgres',
  password: '',
  tables: {
    users: 'users',
    signals: 'trading_signals',
    content: 'content',
    marketData: 'market_data',
    sentiment: 'sentiment_data',
    news: 'news',
    settings: 'settings',
    telegramBots: 'telegram_bots'
  }
}
```

## 🚀 Implementation Steps

### Phase 1: Database Setup (Choose one)

#### Option A: MongoDB
```bash
npm install mongoose
```

#### Option B: PostgreSQL
```bash
npm install pg sequelize
```

### Phase 2: Replace File Storage
1. Update `routes/api.js` to use database queries
2. Implement proper validation using schemas
3. Add authentication middleware for API routes

### Phase 3: Enhanced Features
1. Add pagination to API endpoints
2. Implement search functionality
3. Add real-time notifications
4. Integrate Telegram bot API

## 🔐 Security Considerations

### API Protection
```javascript
// Add to app.js
const jwt = require('jsonwebtoken');

// Protect API routes
app.use('/api', authenticateToken);

function authenticateToken(req, res, next) {
  const authHeader = req.headers['authorization'];
  const token = authHeader && authHeader.split(' ')[1];

  if (!token) {
    return res.sendStatus(401);
  }

  jwt.verify(token, process.env.ACCESS_TOKEN_SECRET, (err, user) => {
    if (err) return res.sendStatus(403);
    req.user = user;
    next();
  });
}
```

### Environment Variables
Create `.env` file:
```
MONGODB_URL=mongodb://localhost:27017/maya-eslami
TELEGRAM_BOT_TOKEN=your_bot_token
TELEGRAM_CHANNEL_ID=your_channel_id
ACCESS_TOKEN_SECRET=your_jwt_secret
DB_HOST=localhost
DB_PORT=5432
DB_NAME=maya_eslami
DB_USER=postgres
DB_PASSWORD=your_password
```

## 📋 Form Field Mapping

### Admin Signal Form → Database
```javascript
// Form fields in admin/signals.ejs map to TradingSignalSchema
type → type
market → market
symbol → symbol (نماد معاملاتی)
entryPrice → entryPrice (قیمت ورود)
stopLoss → stopLoss (حد ضرر)
takeProfit → takeProfit (هدف قیمتی)
analysis → analysis (توضیحات تحلیل)
telegramChannel → telegramChannel
sentAt → sentAt (زمان ارسال)
```

### Admin Content Form → Database
```javascript
// Form fields in admin/content.ejs map to ContentSchema
title → title (عنوان)
category → category (دسته‌بندی)
summary → summary (خلاصه)
content → content (محتوا)
status → status (publishImmediate checkbox → 'published' : 'draft')
```

## 🔄 Migration Notes

1. **Data Migration**: Current JSON files can be imported into chosen database
2. **API Compatibility**: Existing frontend code will work with new API endpoints
3. **Admin Interface**: No changes needed to admin forms - they're already synchronized
4. **Gradual Migration**: Can implement database per module (signals first, then content, etc.)

## 📞 Integration Support

All admin forms now match the exact field requirements for backend integration. The data flows seamlessly from:

```
Admin Form → API Endpoint → Database → Frontend Display
```

The synchronization ensures:
- ✅ No field mismatches
- ✅ Consistent data types
- ✅ Proper validation rules
- ✅ Backend-ready schemas
- ✅ API endpoint compatibility