const express = require('express');
const session = require('express-session');
const path = require('path');
const { startPeriodicScraping } = require('./scraper');
const { startGoldScraper } = require('./gold-scraper');
const { startSentimentScraper } = require('./sentiment-scraper');
const CryptoScraper = require('./crypto-scraper');

const app = express();
const PORT = process.env.PORT || 3000;

app.set('view engine', 'ejs');
app.set('views', path.join(__dirname, 'views'));

// Session configuration
app.use(session({
  secret: 'maya-eslami-admin-secret-key',
  resave: false,
  saveUninitialized: false,
  cookie: {
    secure: false, // Set to true in production with HTTPS
    maxAge: 24 * 60 * 60 * 1000 // 24 hours
  }
}));

app.use(express.static(path.join(__dirname, 'public')));
app.use(express.urlencoded({ extended: true }));
app.use(express.json());

// Import routes and middleware
const adminRoutes = require('./routes/admin');
const apiRoutes = require('./routes/api');
const adminAuth = require('./middleware/adminAuth');

app.get('/', (req, res) => {
  res.render('pages/landing');
});

app.get('/index', (req, res) => {
  res.render('pages/index');
});

app.get('/prices', (req, res) => {
  res.render('pages/prices');
});

app.get('/currency', (req, res) => {
  res.render('pages/currency');
});

app.get('/articles', (req, res) => {
  res.render('pages/articles');
});

app.get('/news', (req, res) => {
  res.render('pages/news');
});

app.get('/subscription', (req, res) => {
  res.render('pages/subscription');
});

app.get('/contact', (req, res) => {
  res.render('pages/contact');
});

app.get('/profile', (req, res) => {
  res.render('pages/profile');
});

app.get('/edit-profile', (req, res) => {
  res.render('pages/edit-profile');
});

app.get('/calendar', (req, res) => {
  res.render('pages/calendar');
});

app.get('/sentiment', (req, res) => {
  res.render('pages/sentiment');
});

app.get('/gold', (req, res) => {
  res.render('pages/gold');
});

app.get('/crypto', (req, res) => {
  res.render('pages/crypto');
});

// Individual signal pages
app.get('/signal/:id', (req, res) => {
  const signalId = req.params.id;
  res.render('pages/signal', { signalId: signalId });
});

app.post('/login', (req, res) => {
  res.redirect('/index');
});

app.post('/register', (req, res) => {
  res.redirect('/index');
});

// API endpoint for currency data
app.get('/api/currencies', (req, res) => {
  const fs = require('fs');
  const currenciesPath = path.join(__dirname, 'public', 'api', 'currencies.json');

  try {
    if (fs.existsSync(currenciesPath)) {
      const data = fs.readFileSync(currenciesPath, 'utf8');
      res.json(JSON.parse(data));
    } else {
      res.status(404).json({ error: 'Currency data not found' });
    }
  } catch (error) {
    res.status(500).json({ error: 'Error reading currency data' });
  }
});

// API endpoint for gold data
app.get('/api/gold', (req, res) => {
  const fs = require('fs');
  const goldPath = path.join(__dirname, 'public', 'api', 'gold.json');

  try {
    if (fs.existsSync(goldPath)) {
      const data = fs.readFileSync(goldPath, 'utf8');
      res.json(JSON.parse(data));
    } else {
      res.status(404).json({ error: 'Gold data not found' });
    }
  } catch (error) {
    res.status(500).json({ error: 'Error reading gold data' });
  }
});

// API endpoint for sentiment data
app.get('/api/sentiment', (req, res) => {
  const fs = require('fs');
  const sentimentPath = path.join(__dirname, 'public', 'api', 'sentiment.json');

  try {
    if (fs.existsSync(sentimentPath)) {
      const data = fs.readFileSync(sentimentPath, 'utf8');
      res.json(JSON.parse(data));
    } else {
      // Return mock sentiment data if file doesn't exist
      const mockData = {
        majorPairs: [
          { pair: 'EUR/USD', bullish: 65, bearish: 35, change: '+2.3%' },
          { pair: 'GBP/USD', bullish: 58, bearish: 42, change: '+1.8%' },
          { pair: 'USD/JPY', bullish: 72, bearish: 28, change: '+3.1%' },
          { pair: 'USD/CHF', bullish: 45, bearish: 55, change: '-1.2%' }
        ],
        commodities: [
          { commodity: 'Gold', bullish: 78, bearish: 22, change: '+4.2%' },
          { commodity: 'Silver', bullish: 68, bearish: 32, change: '+2.8%' },
          { commodity: 'Oil', bullish: 52, bearish: 48, change: '+0.5%' },
          { commodity: 'Bitcoin', bullish: 85, bearish: 15, change: '+8.7%' }
        ],
        lastUpdate: new Date().toISOString()
      };
      res.json(mockData);
    }
  } catch (error) {
    res.status(500).json({ error: 'Error reading sentiment data' });
  }
});

// API endpoint for crypto data
app.get('/api/crypto', (req, res) => {
  const fs = require('fs');
  const cryptoPath = path.join(__dirname, 'public', 'api', 'crypto.json');

  try {
    if (fs.existsSync(cryptoPath)) {
      const data = fs.readFileSync(cryptoPath, 'utf8');
      res.json(JSON.parse(data));
    } else {
      res.status(404).json({ error: 'Crypto data not found' });
    }
  } catch (error) {
    res.status(500).json({ error: 'Error reading crypto data' });
  }
});

// API routes (should be protected in production)
app.use('/api', apiRoutes);

// Admin routes with authentication
app.use('/admin', adminAuth, adminRoutes);

app.listen(PORT, () => {
  console.log(`Server running on http://localhost:${PORT}`);
  console.log(`Admin panel: http://localhost:${PORT}/admin/login`);
  console.log('==================================================');

  // Start all scrapers automatically
  console.log('🚀 Starting all data scrapers...');

  // Start currency scraper
  console.log('💱 Starting currency scraper...');
  startPeriodicScraping();

  // Start gold scraper
  console.log('🥇 Starting gold scraper...');
  startGoldScraper();

  // Start sentiment scraper
  console.log('📊 Starting sentiment scraper...');
  startSentimentScraper();

  // Start crypto scraper
  console.log('💰 Starting crypto scraper...');
  const cryptoScraper = new CryptoScraper();
  cryptoScraper.start();

  console.log('✅ All scrapers started successfully!');
  console.log('📊 Data will be updated automatically:');
  console.log('   - Currency & Gold: Every 1 minute');
  console.log('   - Crypto: Every 1 minute');
  console.log('   - Sentiment: Every 5 minutes');
  console.log('🌐 API endpoints available:');
  console.log(`   - Currency: http://localhost:${PORT}/api/currencies`);
  console.log(`   - Gold: http://localhost:${PORT}/api/gold`);
  console.log(`   - Crypto: http://localhost:${PORT}/api/crypto`);
  console.log(`   - Sentiment: http://localhost:${PORT}/api/sentiment`);
});