// Database configuration for Maya Eslami trading platform
// This file provides database setup for when you're ready to implement backend storage

const path = require('path');
const fs = require('fs');

// File-based storage configuration (for current JSON-based approach)
const FileStorage = {
  dataDir: path.join(__dirname, '..', 'public', 'api'),

  // Ensure data directory exists
  init() {
    if (!fs.existsSync(this.dataDir)) {
      fs.mkdirSync(this.dataDir, { recursive: true });
    }
  },

  // Read data from JSON file
  read(filename) {
    const filePath = path.join(this.dataDir, filename);
    try {
      if (fs.existsSync(filePath)) {
        const data = fs.readFileSync(filePath, 'utf8');
        return JSON.parse(data);
      }
      return null;
    } catch (error) {
      console.error(`Error reading ${filename}:`, error);
      return null;
    }
  },

  // Write data to JSON file
  write(filename, data) {
    const filePath = path.join(this.dataDir, filename);
    try {
      fs.writeFileSync(filePath, JSON.stringify(data, null, 2), 'utf8');
      return true;
    } catch (error) {
      console.error(`Error writing ${filename}:`, error);
      return false;
    }
  }
};

// MongoDB configuration (for future implementation)
const MongoConfig = {
  url: process.env.MONGODB_URL || 'mongodb://localhost:27017/maya-eslami',
  options: {
    useNewUrlParser: true,
    useUnifiedTopology: true,
    maxPoolSize: 10,
    serverSelectionTimeoutMS: 5000,
    socketTimeoutMS: 45000,
  },

  // Collections
  collections: {
    users: 'users',
    signals: 'trading_signals',
    content: 'content',
    marketData: 'market_data',
    sentiment: 'sentiment_data',
    news: 'news',
    settings: 'settings',
    telegramBots: 'telegram_bots'
  }
};

// PostgreSQL configuration (alternative to MongoDB)
const PostgreSQLConfig = {
  host: process.env.DB_HOST || 'localhost',
  port: process.env.DB_PORT || 5432,
  database: process.env.DB_NAME || 'maya_eslami',
  username: process.env.DB_USER || 'postgres',
  password: process.env.DB_PASSWORD || '',
  dialect: 'postgres',
  pool: {
    max: 5,
    min: 0,
    acquire: 30000,
    idle: 10000
  },

  // Tables
  tables: {
    users: 'users',
    signals: 'trading_signals',
    content: 'content',
    marketData: 'market_data',
    sentiment: 'sentiment_data',
    news: 'news',
    settings: 'settings',
    telegramBots: 'telegram_bots'
  }
};

// Database abstraction layer
class DatabaseAdapter {
  constructor(type = 'file') {
    this.type = type;
    this.storage = FileStorage;

    if (type === 'file') {
      this.storage.init();
    }
  }

  // Generic CRUD operations
  async create(collection, data) {
    if (this.type === 'file') {
      const filename = `${collection}.json`;
      const existing = this.storage.read(filename) || [];
      data.id = data.id || Date.now().toString();
      data.createdAt = data.createdAt || new Date();
      existing.push(data);
      return this.storage.write(filename, existing);
    }
    // Add MongoDB/PostgreSQL implementation here
  }

  async findAll(collection, filter = {}) {
    if (this.type === 'file') {
      const filename = `${collection}.json`;
      const data = this.storage.read(filename) || [];

      // Apply basic filtering
      if (Object.keys(filter).length === 0) {
        return data;
      }

      return data.filter(item => {
        return Object.keys(filter).every(key => item[key] === filter[key]);
      });
    }
    // Add MongoDB/PostgreSQL implementation here
  }

  async findById(collection, id) {
    if (this.type === 'file') {
      const filename = `${collection}.json`;
      const data = this.storage.read(filename) || [];
      return data.find(item => item.id === id);
    }
    // Add MongoDB/PostgreSQL implementation here
  }

  async update(collection, id, updateData) {
    if (this.type === 'file') {
      const filename = `${collection}.json`;
      const data = this.storage.read(filename) || [];
      const index = data.findIndex(item => item.id === id);

      if (index !== -1) {
        data[index] = { ...data[index], ...updateData, updatedAt: new Date() };
        return this.storage.write(filename, data);
      }
      return false;
    }
    // Add MongoDB/PostgreSQL implementation here
  }

  async delete(collection, id) {
    if (this.type === 'file') {
      const filename = `${collection}.json`;
      const data = this.storage.read(filename) || [];
      const filtered = data.filter(item => item.id !== id);
      return this.storage.write(filename, filtered);
    }
    // Add MongoDB/PostgreSQL implementation here
  }
}

// Export configurations and adapter
module.exports = {
  FileStorage,
  MongoConfig,
  PostgreSQLConfig,
  DatabaseAdapter
};