const axios = require('axios');
const cheerio = require('cheerio');
const fs = require('fs');
const path = require('path');

class CryptoScraper {
    constructor() {
        this.url = 'https://alanchand.com/crypto-price';
        this.outputPath = path.join(__dirname, 'public', 'api', 'crypto.json');
        this.updateInterval = 60000; // 1 minute
        this.intervalId = null;
    }

    async fetchCryptoData() {
        try {
            console.log('🚀 Starting crypto scraper...');
            console.log('🔄 Fetching crypto data from alanchand.com...');

            const response = await axios.get(this.url, {
                headers: {
                    'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36',
                    'Accept': 'text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
                    'Accept-Language': 'en-US,en;q=0.5',
                    'Accept-Encoding': 'gzip, deflate, br',
                    'Connection': 'keep-alive',
                    'Upgrade-Insecure-Requests': '1'
                },
                timeout: 10000
            });

            const $ = cheerio.load(response.data);
            const cryptoData = [];

            // Find the crypto table and extract data
            $('.cryptoTbl tbody tr').each((index, element) => {
                const row = $(element);

                // Extract crypto name and symbol
                const nameCell = row.find('.currNameIco');
                const faName = nameCell.find('.faCurr').text().trim();
                const enName = nameCell.find('.enCurr').text().trim();
                const symbol = nameCell.find('.symbolCurr').text().trim();
                const iconSrc = nameCell.find('img').attr('src');

                // Extract Toman price and change
                const tomanCell = row.find('.perTmn');
                const tomanPrice = tomanCell.find('.tmn').text().trim().replace(/[,۰-۹]/g, match => {
                    const persianDigits = '۰۱۲۳۴۵۶۷۸۹';
                    const englishDigits = '0123456789';
                    if (match === ',') return '';
                    return englishDigits[persianDigits.indexOf(match)] || match;
                });

                const tomanChangeElement = tomanCell.find('.priceSymbol');
                const tomanChangeText = tomanChangeElement.text().replace('%', '').trim();
                const tomanChangeDirection = tomanChangeElement.hasClass('up') ? 'up' :
                                           tomanChangeElement.hasClass('down') ? 'down' : 'no_change';

                // Extract USD price and change
                const usdCell = row.find('.perUsd');
                const usdPrice = usdCell.find('.dlr').text().trim();
                const usdChangeElement = usdCell.find('.priceSymbol');
                const usdChangeText = usdChangeElement.text().replace('%', '').trim();
                const usdChangeDirection = usdChangeElement.hasClass('up') ? 'up' :
                                         usdChangeElement.hasClass('down') ? 'down' : 'no_change';

                // Only add if we have valid data
                if (faName && symbol && tomanPrice) {
                    const crypto = {
                        faName: faName,
                        enName: enName,
                        symbol: symbol.toUpperCase(),
                        icon: iconSrc ? `https://alanchand.com${iconSrc}` : null,
                        prices: {
                            toman: {
                                value: parseInt(tomanPrice) || 0,
                                formatted: this.addCommas(tomanPrice),
                                change: {
                                    value: tomanChangeText,
                                    direction: tomanChangeDirection
                                }
                            },
                            usd: {
                                value: parseFloat(usdPrice) || 0,
                                formatted: usdPrice,
                                change: {
                                    value: usdChangeText,
                                    direction: usdChangeDirection
                                }
                            }
                        },
                        lastUpdate: new Date().toISOString()
                    };

                    cryptoData.push(crypto);
                    console.log(`✅ ${faName} (${symbol}): ${this.addCommas(tomanPrice)} تومان, $${usdPrice}`);
                }
            });

            if (cryptoData.length > 0) {
                const finalData = {
                    lastUpdate: new Date().toISOString(),
                    totalCryptos: cryptoData.length,
                    data: cryptoData
                };

                // Save to file
                await this.saveToFile(finalData);
                console.log(`💾 Crypto data saved to ${this.outputPath}`);
                console.log(`📊 Total cryptocurrencies: ${cryptoData.length}`);

                return finalData;
            } else {
                console.log('❌ No crypto data found');
                return null;
            }

        } catch (error) {
            console.error('❌ Error fetching crypto data:', error.message);
            return null;
        }
    }

    addCommas(numberString) {
        if (!numberString) return '0';
        return numberString.toString().replace(/\B(?=(\d{3})+(?!\d))/g, ',');
    }

    async saveToFile(data) {
        try {
            // Ensure directory exists
            const dir = path.dirname(this.outputPath);
            if (!fs.existsSync(dir)) {
                fs.mkdirSync(dir, { recursive: true });
            }

            // Write file
            fs.writeFileSync(this.outputPath, JSON.stringify(data, null, 2), 'utf8');
        } catch (error) {
            console.error('❌ Error saving crypto data:', error.message);
        }
    }

    start() {
        console.log('🚀 Starting crypto scraper...');

        // Fetch immediately
        this.fetchCryptoData();

        // Set up interval
        this.intervalId = setInterval(() => {
            console.log('\n⏰ Running scheduled crypto update...');
            this.fetchCryptoData();
        }, this.updateInterval);

        console.log(`⏱️  Crypto scraper will update every ${this.updateInterval / 1000} seconds`);
    }

    stop() {
        if (this.intervalId) {
            clearInterval(this.intervalId);
            this.intervalId = null;
            console.log('🛑 Crypto scraper stopped');
        }
    }
}

module.exports = CryptoScraper;