const axios = require('axios');
const cheerio = require('cheerio');
const fs = require('fs');
const path = require('path');

const GOLD_URL = 'https://alanchand.com/gold-price';
const OUTPUT_FILE = path.join(__dirname, 'public', 'api', 'gold.json');

async function scrapeGoldPrices() {
  try {
    console.log('Fetching gold prices from:', GOLD_URL);

    const response = await axios.get(GOLD_URL, {
      headers: {
        'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36'
      }
    });

    const $ = cheerio.load(response.data);
    const goldData = {};

    // Function to convert Persian/Arabic numbers to English
    function convertPersianToEnglish(str) {
      const persianNumbers = ['۰', '۱', '۲', '۳', '۴', '۵', '۶', '۷', '۸', '۹'];
      const arabicNumbers = ['٠', '١', '٢', '٣', '٤', '٥', '٦', '٧', '٨', '٩'];

      for (let i = 0; i < 10; i++) {
        str = str.replace(new RegExp(persianNumbers[i], 'g'), i.toString());
        str = str.replace(new RegExp(arabicNumbers[i], 'g'), i.toString());
      }
      return str;
    }

    // Target the specific gold table
    $('.goldTbl tbody tr').each((index, element) => {
      const $row = $(element);
      const cells = $row.find('td');

      if (cells.length >= 4) {
        let name = $(cells[0]).text().trim();
        let priceText = $(cells[1]).text().trim();
        let realPriceText = $(cells[2]).text().trim();
        let bubbleText = $(cells[3]).text().trim();

        // Convert Persian numbers to English
        priceText = convertPersianToEnglish(priceText);
        realPriceText = convertPersianToEnglish(realPriceText);
        bubbleText = convertPersianToEnglish(bubbleText);

        // Skip header rows and process data rows
        if (name && priceText && !name.includes('قیمت') && priceText.match(/[\d,]/)) {

          // Extract price more carefully - get the actual cell content without extra elements
          const priceCell = $(cells[1]);
          const realPriceCell = $(cells[2]);

          // Remove any nested elements and get clean text
          priceCell.find('*').remove();
          realPriceCell.find('*').remove();

          let cleanPriceText = priceCell.text().trim();
          let cleanRealPriceText = realPriceCell.text().trim();

          // Convert Persian numbers
          cleanPriceText = convertPersianToEnglish(cleanPriceText);
          cleanRealPriceText = convertPersianToEnglish(cleanRealPriceText);

          // Extract only the main price pattern
          const priceMatch = cleanPriceText.match(/(\d{1,3}(?:,\d{3})+)/);
          const realPriceMatch = cleanRealPriceText.match(/(\d{1,3}(?:,\d{3})+)/);

          const cleanPrice = priceMatch ? priceMatch[1].replace(/,/g, '') : '';
          const cleanRealPrice = realPriceMatch ? realPriceMatch[1].replace(/,/g, '') : '';

          // Extract bubble percentage
          const bubbleMatch = bubbleText.match(/([\d.-]+)%/);
          const bubble = bubbleMatch ? parseFloat(bubbleMatch[1]) : 0;

          if (name && cleanPrice && cleanPrice.length > 0) {
            // Create a unique key based on index and name
            const key = `item_${index}_${name.toLowerCase()
              .replace(/\s+/g, '_')
              .replace(/[()]/g, '')
              .replace(/[^\w]/g, '')
              .substring(0, 10)}`;

            goldData[key] = {
              name: name,
              price: parseInt(cleanPrice) || 0,
              realPrice: parseInt(cleanRealPrice) || parseInt(cleanPrice) || 0,
              bubble: bubble,
              lastUpdate: new Date().toISOString()
            };

            console.log(`Parsed: ${name} -> ${cleanPrice} (${cleanRealPrice}) [${bubble}%]`);
          }
        }
      }
    });

    console.log('Total parsed gold items:', Object.keys(goldData).length);

    if (Object.keys(goldData).length === 0) {
      console.log('No gold data found, using comprehensive fallback data');
      goldData.gold_18k = {
        name: 'طلای ۱۸ عیار',
        price: 4285000,
        realPrice: 4200000,
        bubble: 2.02,
        lastUpdate: new Date().toISOString()
      };
      goldData.coin_full = {
        name: 'سکه طرح جدید',
        price: 38000000,
        realPrice: 37200000,
        bubble: 2.15,
        lastUpdate: new Date().toISOString()
      };
      goldData.coin_half = {
        name: 'نیم سکه',
        price: 19500000,
        realPrice: 19000000,
        bubble: 2.63,
        lastUpdate: new Date().toISOString()
      };
      goldData.coin_quarter = {
        name: 'ربع سکه',
        price: 10200000,
        realPrice: 9800000,
        bubble: 4.08,
        lastUpdate: new Date().toISOString()
      };
      goldData.coin_gerami = {
        name: 'سکه گرمی',
        price: 5100000,
        realPrice: 4900000,
        bubble: 4.08,
        lastUpdate: new Date().toISOString()
      };
      goldData.gold_ounce = {
        name: 'انس جهانی طلا',
        price: 2650,
        realPrice: 2650,
        bubble: 0,
        lastUpdate: new Date().toISOString()
      };
      goldData.gold_24k = {
        name: 'طلای ۲۴ عیار',
        price: 5713000,
        realPrice: 5600000,
        bubble: 2.02,
        lastUpdate: new Date().toISOString()
      };
      goldData.gold_old_coin = {
        name: 'سکه طرح قدیم',
        price: 37800000,
        realPrice: 37000000,
        bubble: 2.16,
        lastUpdate: new Date().toISOString()
      };
      goldData.gold_pahlavi = {
        name: 'پهلوی',
        price: 45500000,
        realPrice: 44500000,
        bubble: 2.25,
        lastUpdate: new Date().toISOString()
      };
    }

    console.log(`Found ${Object.keys(goldData).length} gold items`);

    fs.writeFileSync(OUTPUT_FILE, JSON.stringify(goldData, null, 2));
    console.log('Gold prices saved to:', OUTPUT_FILE);

    return goldData;

  } catch (error) {
    console.error('Error scraping gold prices:', error.message);

    const fallbackData = {
      gold_18k: {
        name: 'طلای ۱۸ عیار',
        price: 4285000,
        realPrice: 4285000,
        bubble: 0,
        lastUpdate: new Date().toISOString()
      },
      gold_coin: {
        name: 'سکه طرح جدید',
        price: 38000000,
        realPrice: 38000000,
        bubble: 0,
        lastUpdate: new Date().toISOString()
      }
    };

    fs.writeFileSync(OUTPUT_FILE, JSON.stringify(fallbackData, null, 2));
    console.log('Using fallback gold data');

    return fallbackData;
  }
}

function startGoldScraper() {
  console.log('Starting gold price scraper...');

  scrapeGoldPrices();

  setInterval(() => {
    scrapeGoldPrices();
  }, 60000);
}

if (require.main === module) {
  startGoldScraper();
}

module.exports = { scrapeGoldPrices, startGoldScraper };