// Backend-compatible data schemas for Maya Eslami trading platform

// Trading Signal Schema
const TradingSignalSchema = {
  id: { type: 'string', required: true },
  type: { type: 'string', enum: ['buy', 'sell'], required: true },
  market: { type: 'string', enum: ['forex', 'crypto', 'gold', 'stock'], required: true },
  symbol: { type: 'string', required: true }, // EUR/USD, BTC/USDT, XAU/USD
  entryPrice: { type: 'number', required: true },
  stopLoss: { type: 'number', required: true },
  takeProfit: { type: 'number', required: true },
  analysis: { type: 'string' }, // Trading analysis description
  status: { type: 'string', enum: ['active', 'completed', 'cancelled'], default: 'active' },
  telegramChannel: { type: 'string', required: true },
  createdAt: { type: 'date', default: Date.now },
  sentAt: { type: 'date' },
  members: { type: 'number', default: 0 }, // Number of members who received signal
  result: {
    profit: { type: 'number' }, // Profit/Loss in percentage
    closePrice: { type: 'number' },
    closeTime: { type: 'date' }
  }
};

// Content/Article Schema
const ContentSchema = {
  id: { type: 'string', required: true },
  title: { type: 'string', required: true },
  slug: { type: 'string', required: true },
  category: {
    type: 'string',
    enum: [
      'آموزش تحلیل تکنیکال',
      'استراتژی معاملاتی',
      'اخبار بازار مالی',
      'مدیریت ریسک',
      'احساسات بازار',
      'تحلیل فارکس',
      'تحلیل کریپتو',
      'تحلیل طلا',
      'اخبار اقتصادی',
      'راهنمای تریدینگ'
    ],
    required: true
  },
  summary: { type: 'string', required: true },
  content: { type: 'string', required: true },
  featuredImage: { type: 'string' },
  status: { type: 'string', enum: ['draft', 'published', 'archived'], default: 'draft' },
  publishedAt: { type: 'date' },
  views: { type: 'number', default: 0 },
  author: { type: 'string', required: true },
  tags: [{ type: 'string' }],
  seoTitle: { type: 'string' },
  seoDescription: { type: 'string' },
  createdAt: { type: 'date', default: Date.now },
  updatedAt: { type: 'date', default: Date.now }
};

// User Schema
const UserSchema = {
  id: { type: 'string', required: true },
  name: { type: 'string', required: true },
  email: { type: 'string', required: true, unique: true },
  password: { type: 'string', required: true }, // Should be hashed
  role: { type: 'string', enum: ['user', 'premium', 'admin'], default: 'user' },
  subscription: {
    type: { type: 'string', enum: ['free', 'monthly', 'yearly'], default: 'free' },
    startDate: { type: 'date' },
    endDate: { type: 'date' },
    status: { type: 'string', enum: ['active', 'expired', 'cancelled'], default: 'active' }
  },
  telegramUserId: { type: 'string' },
  preferences: {
    language: { type: 'string', default: 'fa' },
    notifications: { type: 'boolean', default: true },
    markets: [{ type: 'string' }] // Preferred markets: forex, crypto, gold, stock
  },
  createdAt: { type: 'date', default: Date.now },
  lastLogin: { type: 'date' },
  isActive: { type: 'boolean', default: true }
};

// Market Data Schema (for currency, gold, etc.)
const MarketDataSchema = {
  id: { type: 'string', required: true },
  symbol: { type: 'string', required: true }, // USD/IRR, EUR/USD, XAU/USD, BTC/USDT
  name: { type: 'string', required: true },
  type: { type: 'string', enum: ['currency', 'gold', 'crypto', 'stock'], required: true },
  price: { type: 'number', required: true },
  change: { type: 'number' }, // Change in value
  changePercent: { type: 'number' }, // Change in percentage
  high24h: { type: 'number' },
  low24h: { type: 'number' },
  volume: { type: 'number' },
  lastUpdate: { type: 'date', default: Date.now },
  source: { type: 'string' } // Data source
};

// Sentiment Data Schema
const SentimentSchema = {
  id: { type: 'string', required: true },
  symbol: { type: 'string', required: true },
  type: { type: 'string', enum: ['currency', 'commodity', 'crypto'], required: true },
  bullish: { type: 'number', min: 0, max: 100, required: true }, // Bullish percentage
  bearish: { type: 'number', min: 0, max: 100, required: true }, // Bearish percentage
  change: { type: 'string' }, // Change indicator like +2.3%
  lastUpdate: { type: 'date', default: Date.now },
  source: { type: 'string' }
};

// News Schema
const NewsSchema = {
  id: { type: 'string', required: true },
  title: { type: 'string', required: true },
  summary: { type: 'string' },
  content: { type: 'string' },
  category: {
    type: 'string',
    enum: ['forex', 'crypto', 'gold', 'economy', 'politics', 'central-bank'],
    required: true
  },
  impact: { type: 'string', enum: ['low', 'medium', 'high'] }, // Market impact
  affectedCurrencies: [{ type: 'string' }], // Currencies that might be affected
  publishedAt: { type: 'date', required: true },
  source: { type: 'string' },
  imageUrl: { type: 'string' },
  externalUrl: { type: 'string' },
  createdAt: { type: 'date', default: Date.now }
};

// Settings Schema (for admin configuration)
const SettingsSchema = {
  id: { type: 'string', required: true },
  category: { type: 'string', required: true }, // telegram, general, api, etc.
  key: { type: 'string', required: true },
  value: { type: 'string', required: true },
  description: { type: 'string' },
  type: { type: 'string', enum: ['string', 'number', 'boolean', 'json'], default: 'string' },
  updatedAt: { type: 'date', default: Date.now },
  updatedBy: { type: 'string' } // Admin user ID
};

// Telegram Bot Configuration Schema
const TelegramBotSchema = {
  id: { type: 'string', required: true },
  botToken: { type: 'string', required: true },
  channelId: { type: 'string', required: true },
  channelUsername: { type: 'string' },
  isActive: { type: 'boolean', default: true },
  subscriberCount: { type: 'number', default: 0 },
  lastMessageId: { type: 'string' },
  createdAt: { type: 'date', default: Date.now }
};

module.exports = {
  TradingSignalSchema,
  ContentSchema,
  UserSchema,
  MarketDataSchema,
  SentimentSchema,
  NewsSchema,
  SettingsSchema,
  TelegramBotSchema
};