class ThemeManager {
  constructor() {
    this.init();
  }

  init() {
    this.loadTheme();
    this.bindEvents();
    this.updateIcons();
    this.initPersianDate();
  }

  loadTheme() {
    const savedTheme = localStorage.getItem('theme') || 'light';
    this.setTheme(savedTheme);
  }

  setTheme(theme) {
    if (theme === 'dark') {
      document.documentElement.classList.add('dark');
    } else {
      document.documentElement.classList.remove('dark');
    }
    localStorage.setItem('theme', theme);
    this.updateIcons();
  }

  toggleTheme() {
    const isDark = document.documentElement.classList.contains('dark');
    this.setTheme(isDark ? 'light' : 'dark');
  }

  updateIcons() {
    const isDark = document.documentElement.classList.contains('dark');

    // Desktop icons
    const lightIcon = document.getElementById('theme-toggle-light-icon');
    const darkIcon = document.getElementById('theme-toggle-dark-icon');

    if (lightIcon && darkIcon) {
      if (isDark) {
        lightIcon.classList.remove('hidden');
        darkIcon.classList.add('hidden');
      } else {
        lightIcon.classList.add('hidden');
        darkIcon.classList.remove('hidden');
      }
    }

    // Mobile icons
    const mobileLightIcon = document.getElementById('mobile-theme-toggle-light-icon');
    const mobileDarkIcon = document.getElementById('mobile-theme-toggle-dark-icon');

    if (mobileLightIcon && mobileDarkIcon) {
      if (isDark) {
        mobileLightIcon.classList.remove('hidden');
        mobileDarkIcon.classList.add('hidden');
      } else {
        mobileLightIcon.classList.add('hidden');
        mobileDarkIcon.classList.remove('hidden');
      }
    }
  }

  bindEvents() {
    const themeToggle = document.getElementById('theme-toggle');
    if (themeToggle) {
      themeToggle.addEventListener('click', () => this.toggleTheme());
    }

    const mobileThemeToggle = document.getElementById('mobile-theme-toggle');
    if (mobileThemeToggle) {
      mobileThemeToggle.addEventListener('click', () => this.toggleTheme());
    }

    document.addEventListener('keydown', (e) => {
      if (e.ctrlKey && e.key === 'k') {
        e.preventDefault();
        this.toggleTheme();
      }
    });
  }

  initPersianDate() {
    this.updatePersianDate();
    setInterval(() => this.updatePersianDate(), 60000);
  }

  gregorianToJalali(gy, gm, gd) {
    // More reliable Jalali conversion algorithm
    const g_d_m = [0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334];

    let gy2;
    if (gm > 2) {
      gy2 = gy + 1;
    } else {
      gy2 = gy;
    }

    const days = 365 * gy + Math.floor((gy2 + 3) / 4) - Math.floor((gy2 + 99) / 100) +
                 Math.floor((gy2 + 399) / 400) - 80 + gd + g_d_m[gm - 1];

    let jy = -1029 + 33 * Math.floor(days / 12053);
    let yd = days % 12053;
    yd += 1029 - 33 * Math.floor(yd / 12053);

    let jd;
    if (yd < 366) {
      jy += 0;
      jd = yd;
    } else {
      jy += Math.floor((yd - 366) / 365) + 1;
      jd = (yd - 366) % 365 + 1;
    }

    let jm;
    if (jd < 186) {
      jm = 1 + Math.floor(jd / 31);
      jd = 1 + (jd % 31);
    } else {
      jm = 7 + Math.floor((jd - 186) / 30);
      jd = 1 + ((jd - 186) % 30);
    }

    return [jy, jm, jd];
  }

  getPersianMonthName(month) {
    const months = [
      'فروردین', 'اردیبهشت', 'خرداد', 'تیر', 'مرداد', 'شهریور',
      'مهر', 'آبان', 'آذر', 'دی', 'بهمن', 'اسفند'
    ];
    return months[month - 1];
  }

  getPersianWeekday(date) {
    const weekdays = ['یکشنبه', 'دوشنبه', 'سه‌شنبه', 'چهارشنبه', 'پنج‌شنبه', 'جمعه', 'شنبه'];
    return weekdays[date.getDay()];
  }

  updatePersianDate() {
    const now = new Date();
    const [jy, jm, jd] = this.gregorianToJalali(now.getFullYear(), now.getMonth() + 1, now.getDate());
    const weekday = this.getPersianWeekday(now);
    const monthName = this.getPersianMonthName(jm);

    const persianDate = `${weekday} ${jd} ${monthName}`;

    // Update desktop date
    const desktopPersianDateElement = document.getElementById('desktop-persian-date');
    if (desktopPersianDateElement) {
      desktopPersianDateElement.textContent = persianDate;
    }

    // Update mobile header date
    const mobilePersianDateElement = document.getElementById('mobile-persian-date');
    if (mobilePersianDateElement) {
      mobilePersianDateElement.textContent = persianDate;
    }
  }
}

document.addEventListener('DOMContentLoaded', () => {
  new ThemeManager();
});