const express = require('express');
const router = express.Router();
const fs = require('fs');
const path = require('path');

// Helper function to read JSON data
const readJsonData = (filename) => {
  const filePath = path.join(__dirname, '..', 'public', 'api', filename);
  try {
    if (fs.existsSync(filePath)) {
      const data = fs.readFileSync(filePath, 'utf8');
      return JSON.parse(data);
    }
    return null;
  } catch (error) {
    console.error(`Error reading ${filename}:`, error);
    return null;
  }
};

// Helper function to write JSON data
const writeJsonData = (filename, data) => {
  const filePath = path.join(__dirname, '..', 'public', 'api', filename);
  try {
    fs.writeFileSync(filePath, JSON.stringify(data, null, 2), 'utf8');
    return true;
  } catch (error) {
    console.error(`Error writing ${filename}:`, error);
    return false;
  }
};

// ==================== TRADING SIGNALS API ====================

// Get all trading signals
router.get('/signals', (req, res) => {
  // Mock data for now - replace with actual database query
  const signals = [
    {
      id: '1',
      type: 'buy',
      market: 'forex',
      symbol: 'EUR/USD',
      entryPrice: 1.0850,
      stopLoss: 1.0800,
      takeProfit: 1.0920,
      analysis: 'تحلیل تکنیکال نشان‌دهنده روند صعودی است',
      status: 'active',
      telegramChannel: 'https://t.me/mayaeslami_signals',
      createdAt: new Date(),
      members: 1247
    },
    {
      id: '2',
      type: 'sell',
      market: 'crypto',
      symbol: 'BTC/USDT',
      entryPrice: 67500,
      stopLoss: 69000,
      takeProfit: 65000,
      analysis: 'مقاومت قوی در ناحیه فعلی مشاهده می‌شود',
      status: 'active',
      telegramChannel: 'https://t.me/mayaeslami_signals',
      createdAt: new Date(),
      members: 892
    }
  ];
  res.json(signals);
});

// Create new trading signal
router.post('/signals', (req, res) => {
  const {
    type,
    market,
    symbol,
    entryPrice,
    stopLoss,
    takeProfit,
    analysis,
    telegramChannel
  } = req.body;

  // Validate required fields
  if (!type || !market || !symbol || !entryPrice || !stopLoss || !takeProfit) {
    return res.status(400).json({ error: 'Missing required fields' });
  }

  const newSignal = {
    id: Date.now().toString(),
    type,
    market,
    symbol,
    entryPrice: parseFloat(entryPrice),
    stopLoss: parseFloat(stopLoss),
    takeProfit: parseFloat(takeProfit),
    analysis: analysis || '',
    status: 'active',
    telegramChannel: telegramChannel || 'https://t.me/mayaeslami_signals',
    createdAt: new Date(),
    members: 0
  };

  // In production, save to database
  // For now, return the created signal
  res.status(201).json(newSignal);
});

// Update trading signal
router.put('/signals/:id', (req, res) => {
  const { id } = req.params;
  const updateData = req.body;

  // In production, update in database
  res.json({ id, ...updateData, updatedAt: new Date() });
});

// Delete trading signal
router.delete('/signals/:id', (req, res) => {
  const { id } = req.params;

  // In production, delete from database
  res.json({ message: 'Signal deleted successfully', id });
});

// ==================== CONTENT MANAGEMENT API ====================

// Get all content/articles
router.get('/content', (req, res) => {
  const { category, status } = req.query;

  // Mock data for now
  let articles = [
    {
      id: '1',
      title: 'آموزش تحلیل تکنیکال',
      category: 'آموزش تحلیل تکنیکال',
      summary: 'راهنمای کامل تحلیل تکنیکال برای مبتدیان',
      status: 'published',
      views: 1247,
      author: 'مایا اسلامی',
      createdAt: new Date('2024-03-01'),
      publishedAt: new Date('2024-03-01')
    },
    {
      id: '2',
      title: 'مدیریت ریسک در معاملات',
      category: 'مدیریت ریسک',
      summary: 'نحوه مدیریت ریسک و حفظ سرمایه',
      status: 'draft',
      views: 0,
      author: 'مایا اسلامی',
      createdAt: new Date('2024-02-25')
    }
  ];

  // Filter by category if provided
  if (category && category !== 'همه دسته‌ها') {
    articles = articles.filter(article => article.category === category);
  }

  // Filter by status if provided
  if (status) {
    articles = articles.filter(article => article.status === status);
  }

  res.json(articles);
});

// Create new article
router.post('/content', (req, res) => {
  const {
    title,
    category,
    summary,
    content,
    status = 'draft'
  } = req.body;

  if (!title || !category || !content) {
    return res.status(400).json({ error: 'Missing required fields' });
  }

  const newArticle = {
    id: Date.now().toString(),
    title,
    slug: title.toLowerCase().replace(/\s+/g, '-'),
    category,
    summary: summary || '',
    content,
    status,
    views: 0,
    author: 'Admin',
    createdAt: new Date(),
    updatedAt: new Date()
  };

  if (status === 'published') {
    newArticle.publishedAt = new Date();
  }

  res.status(201).json(newArticle);
});

// ==================== USER MANAGEMENT API ====================

// Get all users
router.get('/users', (req, res) => {
  // Mock data
  const users = [
    {
      id: '1',
      name: 'علی احمدی',
      email: 'ali@example.com',
      role: 'premium',
      subscription: {
        type: 'monthly',
        status: 'active',
        endDate: new Date('2024-12-31')
      },
      createdAt: new Date('2024-01-15'),
      lastLogin: new Date(),
      isActive: true
    },
    {
      id: '2',
      name: 'سارا محمدی',
      email: 'sara@example.com',
      role: 'user',
      subscription: {
        type: 'free',
        status: 'active'
      },
      createdAt: new Date('2024-02-10'),
      lastLogin: new Date(),
      isActive: true
    }
  ];

  res.json(users);
});

// ==================== MARKET DATA API ====================

// Get current market data
router.get('/market-data', (req, res) => {
  const { type } = req.query; // currency, gold, crypto

  const currencies = readJsonData('currencies.json');
  const gold = readJsonData('gold.json');
  const sentiment = readJsonData('sentiment.json');

  let data = {};

  if (!type || type === 'currency') {
    data.currencies = currencies;
  }

  if (!type || type === 'gold') {
    data.gold = gold;
  }

  if (!type || type === 'sentiment') {
    data.sentiment = sentiment;
  }

  res.json(data);
});

// ==================== SETTINGS API ====================

// Get all settings
router.get('/settings', (req, res) => {
  const settings = {
    telegram: {
      botToken: process.env.TELEGRAM_BOT_TOKEN || '',
      channelId: process.env.TELEGRAM_CHANNEL_ID || '',
      channelUsername: '@mayaeslami_signals'
    },
    general: {
      siteName: 'مایا اسلامی',
      adminEmail: 'admin@mayaeslami.com',
      timezone: 'Asia/Tehran'
    },
    api: {
      updateInterval: 60, // seconds
      maxRetries: 3
    }
  };

  res.json(settings);
});

// Update settings
router.put('/settings', (req, res) => {
  const { category, settings } = req.body;

  // In production, save to database
  res.json({
    message: 'Settings updated successfully',
    category,
    updatedAt: new Date()
  });
});

// ==================== ANALYTICS API ====================

// Get dashboard analytics
router.get('/analytics/dashboard', (req, res) => {
  const analytics = {
    totalUsers: 1247,
    activeSignals: 23,
    monthlyRevenue: 12345678, // In Toman
    successRate: 87.3,
    recentActivity: [
      {
        type: 'user_signup',
        user: 'علی احمدی',
        time: new Date()
      },
      {
        type: 'signal_created',
        signal: 'خرید BTCUSDT',
        time: new Date()
      }
    ]
  };

  res.json(analytics);
});

module.exports = router;