const axios = require('axios');
const cheerio = require('cheerio');
const fs = require('fs');
const path = require('path');

// Currency mapping for Persian names to symbols
const currencyMapping = {
    'دلار آمریکا': 'usd',
    'دلار استانبول': 'usd-ist',
    'یورو': 'eur',
    'درهم': 'aed',
    'لیر ترکیه': 'try',
    'پوند انگلیس': 'gbp',
    'یوان چین': 'cny',
    'دلار کانادا': 'cad',
    'دلار استرالیا': 'aud',
    'روبل روسیه': 'rub',
    'صد دینار عراق': 'iqd',
    'رینگیت مالزی': 'myr',
    'لاری گرجستان': 'gel',
    'منات آذربایجان': 'azn',
    'صد درام ارمنستان': 'amd',
    'بات تایلند': 'thb',
    'ریال عمان': 'omr',
    'روپیه هند': 'inr',
    'صد ین ژاپن': 'jpy',
    'افغانی': 'afn'
};

// Function to convert Persian numbers to English
function persianToEnglish(str) {
    const persianNumbers = ['۰', '۱', '۲', '۳', '۴', '۵', '۶', '۷', '۸', '۹'];
    const englishNumbers = ['0', '1', '2', '3', '4', '5', '6', '7', '8', '9'];

    let result = str;
    for (let i = 0; i < persianNumbers.length; i++) {
        result = result.replace(new RegExp(persianNumbers[i], 'g'), englishNumbers[i]);
    }
    return result;
}

// Function to clean and parse price
function parsePrice(priceText) {
    if (!priceText || priceText === '-') return null;

    // Remove commas and convert Persian numbers to English
    const cleaned = persianToEnglish(priceText.replace(/,/g, ''));
    const parsed = parseFloat(cleaned);

    return isNaN(parsed) ? null : parsed;
}

// Function to determine price trend
function getTrend(element) {
    const symbolElement = element.find('.priceSymbol');
    if (symbolElement.hasClass('up')) return 'up';
    if (symbolElement.hasClass('down')) return 'down';
    return 'no_change';
}

// Function to scrape currency data
async function scrapeCurrencyData() {
    try {
        console.log('🔄 Fetching data from alanchand.com...');

        const response = await axios.get('https://alanchand.com/', {
            headers: {
                'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36'
            }
        });

        const $ = cheerio.load(response.data);
        const currencies = {};

        // Find all currency tables
        $('.CurrencyTbl tbody tr').each((index, element) => {
            const row = $(element);

            // Extract currency name
            const nameCell = row.find('.currName');
            const fullName = nameCell.text().trim();

            // Get currency symbol from mapping
            const symbol = currencyMapping[fullName];
            if (!symbol) {
                console.log(`⚠️  Unknown currency: ${fullName}`);
                return; // Skip unknown currencies
            }

            // Extract prices
            const buyPriceText = row.find('.buyPrice').text().trim();
            const sellPriceText = row.find('.sellPrice').clone().children().remove().end().text().trim();
            const usdRateText = row.find('.usdRate').clone().children().remove().end().text().trim();

            // Parse prices
            const buyPrice = parsePrice(buyPriceText);
            const sellPrice = parsePrice(sellPriceText);
            const usdRate = parsePrice(usdRateText);

            // Get trend
            const sellTrend = getTrend(row.find('.sellPrice'));
            const usdTrend = getTrend(row.find('.usdRate'));

            // Calculate change percentage if possible
            let change = 0;
            if (buyPrice && sellPrice) {
                change = ((sellPrice - buyPrice) / buyPrice * 100).toFixed(2);
            }

            // Store currency data
            currencies[symbol] = {
                name: fullName,
                symbol: symbol.toUpperCase(),
                buy: buyPrice,
                sell: sellPrice,
                usdRate: usdRate,
                change: parseFloat(change),
                sellTrend: sellTrend,
                usdTrend: usdTrend,
                lastUpdate: new Date().toISOString()
            };

            console.log(`✅ ${fullName}: Buy ${buyPrice}, Sell ${sellPrice}`);
        });

        // Add gold and coin data (you can customize these or scrape from another source)
        currencies.gold = {
            name: 'طلای ۱۸ عیار',
            symbol: 'GOLD',
            buy: 4285000,
            sell: 4315000,
            change: 2.1,
            lastUpdate: new Date().toISOString()
        };

        currencies.coin = {
            name: 'سکه طرح جدید',
            symbol: 'COIN',
            buy: 38000000,
            sell: 38500000,
            change: 1.8,
            lastUpdate: new Date().toISOString()
        };

        // Save to JSON file
        const jsonPath = path.join(__dirname, 'public', 'api', 'currencies.json');

        // Create api directory if it doesn't exist
        const apiDir = path.dirname(jsonPath);
        if (!fs.existsSync(apiDir)) {
            fs.mkdirSync(apiDir, { recursive: true });
        }

        fs.writeFileSync(jsonPath, JSON.stringify(currencies, null, 2));

        console.log(`💾 Currency data saved to ${jsonPath}`);
        console.log(`📊 Total currencies: ${Object.keys(currencies).length}`);

        return currencies;

    } catch (error) {
        console.error('❌ Error scraping currency data:', error.message);
        throw error;
    }
}

// Function to start periodic scraping
function startPeriodicScraping() {
    console.log('🚀 Starting currency scraper...');

    // Initial scrape
    scrapeCurrencyData().catch(console.error);

    // Set up interval to scrape every minute (60000 ms)
    setInterval(() => {
        console.log('\n⏰ Running scheduled currency update...');
        scrapeCurrencyData().catch(console.error);
    }, 60000);

    console.log('⏱️  Scraper will update every 1 minute');
}

// Export functions
module.exports = {
    scrapeCurrencyData,
    startPeriodicScraping
};

// Run scraper if this file is executed directly
if (require.main === module) {
    startPeriodicScraping();
}