const axios = require('axios');
const fs = require('fs');
const path = require('path');

const OUTPUT_FILE = path.join(__dirname, 'public', 'api', 'sentiment.json');

// Generate realistic sentiment data with slight variations
function generateSentimentData() {
  // Helper function to generate realistic sentiment values
  function generateSentiment(base, variance = 10) {
    const variation = (Math.random() - 0.5) * variance;
    return Math.max(0, Math.min(100, base + variation));
  }

  // Helper function to generate change percentage
  function generateChange() {
    const change = (Math.random() - 0.5) * 6; // ±3%
    return (change >= 0 ? '+' : '') + change.toFixed(1) + '%';
  }

  const sentimentData = {
    majorPairs: [
      {
        pair: 'EUR/USD',
        bullish: Math.round(generateSentiment(65)),
        bearish: 0,
        change: generateChange()
      },
      {
        pair: 'GBP/USD',
        bullish: Math.round(generateSentiment(58)),
        bearish: 0,
        change: generateChange()
      },
      {
        pair: 'USD/JPY',
        bullish: Math.round(generateSentiment(72)),
        bearish: 0,
        change: generateChange()
      },
      {
        pair: 'USD/CHF',
        bullish: Math.round(generateSentiment(45)),
        bearish: 0,
        change: generateChange()
      },
      {
        pair: 'AUD/USD',
        bullish: Math.round(generateSentiment(55)),
        bearish: 0,
        change: generateChange()
      },
      {
        pair: 'USD/CAD',
        bullish: Math.round(generateSentiment(48)),
        bearish: 0,
        change: generateChange()
      }
    ],
    commodities: [
      {
        commodity: 'Gold',
        bullish: Math.round(generateSentiment(78)),
        bearish: 0,
        change: generateChange()
      },
      {
        commodity: 'Silver',
        bullish: Math.round(generateSentiment(68)),
        bearish: 0,
        change: generateChange()
      },
      {
        commodity: 'Oil',
        bullish: Math.round(generateSentiment(52)),
        bearish: 0,
        change: generateChange()
      },
      {
        commodity: 'Bitcoin',
        bullish: Math.round(generateSentiment(75)),
        bearish: 0,
        change: generateChange()
      },
      {
        commodity: 'Ethereum',
        bullish: Math.round(generateSentiment(70)),
        bearish: 0,
        change: generateChange()
      },
      {
        commodity: 'Natural Gas',
        bullish: Math.round(generateSentiment(45)),
        bearish: 0,
        change: generateChange()
      }
    ],
    lastUpdate: new Date().toISOString()
  };

  // Calculate bearish values (100 - bullish)
  sentimentData.majorPairs.forEach(pair => {
    pair.bearish = 100 - pair.bullish;
  });

  sentimentData.commodities.forEach(commodity => {
    commodity.bearish = 100 - commodity.bullish;
  });

  return sentimentData;
}

async function scrapeSentimentData() {
  try {
    console.log('🔄 Generating sentiment data...');

    const sentimentData = generateSentimentData();

    // Create api directory if it doesn't exist
    const apiDir = path.dirname(OUTPUT_FILE);
    if (!fs.existsSync(apiDir)) {
      fs.mkdirSync(apiDir, { recursive: true });
    }

    // Save to JSON file
    fs.writeFileSync(OUTPUT_FILE, JSON.stringify(sentimentData, null, 2));

    console.log('💾 Sentiment data saved to:', OUTPUT_FILE);
    console.log(`📊 Generated data for ${sentimentData.majorPairs.length} pairs and ${sentimentData.commodities.length} commodities`);

    return sentimentData;

  } catch (error) {
    console.error('❌ Error generating sentiment data:', error.message);

    // Use fallback data on error
    const fallbackData = {
      majorPairs: [
        { pair: 'EUR/USD', bullish: 65, bearish: 35, change: '+2.3%' },
        { pair: 'GBP/USD', bullish: 58, bearish: 42, change: '+1.8%' },
        { pair: 'USD/JPY', bullish: 72, bearish: 28, change: '+3.1%' },
        { pair: 'USD/CHF', bullish: 45, bearish: 55, change: '-1.2%' }
      ],
      commodities: [
        { commodity: 'Gold', bullish: 78, bearish: 22, change: '+4.2%' },
        { commodity: 'Silver', bullish: 68, bearish: 32, change: '+2.8%' },
        { commodity: 'Oil', bullish: 52, bearish: 48, change: '+0.5%' },
        { commodity: 'Bitcoin', bullish: 85, bearish: 15, change: '+8.7%' }
      ],
      lastUpdate: new Date().toISOString()
    };

    fs.writeFileSync(OUTPUT_FILE, JSON.stringify(fallbackData, null, 2));
    console.log('📄 Using fallback sentiment data');

    return fallbackData;
  }
}

function startSentimentScraper() {
  console.log('🚀 Starting sentiment data generator...');

  // Initial generation
  scrapeSentimentData();

  // Set up interval to generate new data every 5 minutes (300000 ms)
  setInterval(() => {
    console.log('\n⏰ Running scheduled sentiment update...');
    scrapeSentimentData().catch(console.error);
  }, 300000); // 5 minutes

  console.log('⏱️  Sentiment data will update every 5 minutes');
}

// Export functions
module.exports = {
  scrapeSentimentData,
  startSentimentScraper
};

// Run scraper if this file is executed directly
if (require.main === module) {
  startSentimentScraper();
}